﻿using System;
using System.Threading.Tasks;
using System.Collections.Generic;
using System.Text;

using Xamarin.Forms;

// SII SDK namespace
using SII.SDK.Xamarin.Printer;

namespace Sample
{
    public class Function : IDisposable
    {
        // Define completion delegate of StartDiscoveryDevice
        public event Action<DeviceInfo[]> DeviceSearchCompletedEvent;

        // PrinterManager class
        private PrinterManager _printerManager;

        // Offline status(0x80000000) definition
        private const int STATUS_OFFLINE = -2147483648;

        // Sample print data definition
        private const string RECEIPT_SAMPLE =
        "--------------------------------\n" +
        "GRILLED CHICKEN BREAST   $ 18.50\n" +
        "SIRLOIN STEAK            $ 32.00\n" +
        "ROAST LAMB               $ 20.00\n" +
        "SALAD                    $ 10.00\n" +
        "COKE                     $  3.50\n" +
        "COKE                     $  3.50\n" +
        "ICE CREAM                $  5.00\n" +
        "CHINESE NOODLE           $ 15.00\n" +
        "SUKIYAKI                 $ 30.00\n" +
        "SANDWICH                 $ 10.00\n" +
        "PIZZA                    $ 20.00\n" +
        "TEA                      $  3.50\n" +
        "COFFEE                   $  3.50\n\n" +
        "--------------------------------\n" +
        "       SUBTOTAL        $ 174.50\n" +
        "       SALES TAX       $   8.73\n" +
        "       TOTAL           $ 183.23\n\n" +
        "Thank you and see you again!\n";

        public Function()
        {
            var platformSpecifics = DependencyService.Get<IPlatformSpecifics>();
            _printerManager = new PrinterManager(platformSpecifics.GetApplicationContext());
        }

        public void Dispose()
        {
            if(_printerManager != null)
            {
                if (_printerManager.IsConnect)
                {
                    _printerManager.Disconnect();
                }
                _printerManager = null;
            }
        }

        // Sample of Connect
        public Task<string> ConnectSampleAsync(DeviceModel deviceModel, DeviceInfo deviceInfo)
        {
            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    _printerManager.Connect(deviceModel, deviceInfo);
                }
                catch (PrinterException ex)
                {
                    result = $"Connect Error : {ex.HResult}";
                }

                return result;
            });
        }

        // Sample of Disconnect
        public Task<string> DisonnectSampleAsync()
        {
            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    _printerManager.Disconnect();
                }
                catch (PrinterException ex)
                {
                    result = $"Disconnect Error : {ex.HResult}";
                }

                return result;
            });
        }

        // Sample of printing in standard mode
        public Task<string> StandardModePrintSampleAsync()
        {
            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    // Check offline
                    _printerManager.GetStatus(out int[] status);
                    if (status[0] == STATUS_OFFLINE)
                    {
                        return result = $"SamplePrint Error (OFFLINE)";
                    }

                    // Character set for English
                    _printerManager.CodePage = CodePage.CODE_PAGE_1252;
                    _printerManager.InternationalCharacter = InternationalCharacter.COUNTRY_USA;

                    //// Character set for Japanese
                    //_printerManager.CodePage = CodePage.CODE_PAGE_KATAKANA;
                    //_printerManager.InternationalCharacter = InternationalCharacter.COUNTRY_JAPAN;

                    // Print Image
                    _printerManager.SendDataFile(GetSampleLogoFilePath("SampleLogo.jpg"), PrintAlignment.ALIGNMENT_CENTER, Dithering.DITHERING_ERRORDIFFUSION);

                    // Print Text
                    _printerManager.SendTextEx(RECEIPT_SAMPLE, CharacterBold.BOLD_CANCEL, CharacterUnderline.UNDERLINE_CANCEL, CharacterReverse.REVERSE_CANCEL, CharacterFont.FONT_A, CharacterScale.VERTICAL_1_HORIZONTAL_1, PrintAlignment.ALIGNMENT_CENTER);

                    // Print QR
                    // string qrData = "QR サンプル"; // 2-Byte Character Set example
                    string qrData = "QR Sample";      // 1-Byte Character Set example
                    _printerManager.PrintQRcode(qrData, ErrorCorrection.QR_ERROR_CORRECTION_M, ModuleSize.QR_MODULE_SIZE_4, PrintAlignment.ALIGNMENT_CENTER, QrModel.QR_MODEL_2);

                    // Specify the barcode for EAN13
                    _printerManager.PrintBarcode(BarcodeSymbol.BARCODE_EAN13, "0123456789012", ModuleSize.BARCODE_MODULE_WIDTH_3, 80, HriPosition.HRI_POSITION_BELOW, CharacterFont.FONT_A, PrintAlignment.ALIGNMENT_CENTER);

                    //// Specify the barcode for CODE93 (Data:12345)
                    //// How to binary input.
                    //// 1. Input any data from 00H to 2EH. Multiple data can be input
                    //// 2. Input 2FH or more lastly as the stop code
                    //byte[] data = new byte[] { 0x01, 0x02, 0x03, 0x04, 0x05, 0x2F };
                    //_printerManager.PrintBarcode(BarcodeSymbol.BARCODE_CODE93, data, ModuleSize.BARCODE_MODULE_WIDTH_3, 80, HriPosition.HRI_POSITION_BELOW, CharacterFont.FONT_A, PrintAlignment.ALIGNMENT_CENTER);

                    // Partial cut with pre-feed
                    // If using printer does not have cutter, it will only feed paper
                    _printerManager.CutPaper(CuttingMethod.CUT_PARTIAL);

                    // Confirm print finished
                    _printerManager.GetPrinterResponse(PrinterResponseId.PRINTER_RESPONSE_REQUEST, 0, out int[] response);
                    if (response != null && response.Length != 0)
                    {
                        if (response[0] != (int)0x80)
                        {
                            return result = "StandardModePrintSample Error";
                        }
                    }
                }
                catch (PrinterException ex)
                {
                    result = $"StandardModePrintSample Error : {ex.HResult}";
                }

                return result;
            });
        }

        // Sample of printing in page mode
        public Task<string> PageModePrintSampleAsync()
        {
            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    // Check offline
                    _printerManager.GetStatus(out int[] status);
                    if (status[0] == STATUS_OFFLINE)
                    {
                        return result = $"PageModePrintSample Error (OFFLINE)";
                    }

                    // Character set for English
                    _printerManager.CodePage = CodePage.CODE_PAGE_1252;
                    _printerManager.InternationalCharacter = InternationalCharacter.COUNTRY_USA;

                    //// Character set for Japanese
                    //_printerManager.CodePage = CodePage.CODE_PAGE_KATAKANA;
                    //_printerManager.InternationalCharacter = InternationalCharacter.COUNTRY_JAPAN;

                    // Start page mode
                    _printerManager.EnterPageMode();

                    // Specify the print area in page mode
                    // The number of printable dots in one dot line has to be set to 576 dots for this sample code.
                    _printerManager.SetPageModeArea(0, 0, 576, 355);

                    // Specify the rectangle and the rule
                    _printerManager.PrintPageModeRectangle(0, 0, 575, 344, LineStyle.LINESTYLE_THIN);
                    _printerManager.PrintPageModeRectangle(7, 7, 567, 336, LineStyle.LINESTYLE_THIN);
                    _printerManager.PrintPageModeLine(404, 11, 404, 334, LineStyle.LINESTYLE_THIN);

                    // Specify the text
                    _printerManager.PrintPageModeText(21, 37, "No.123456789");
                    _printerManager.PrintPageModeText(212, 330, "Date 2020-01-01");

                    // Specify the image file
                    string imagePath = GetSampleLogoFilePath("SampleLogo.jpg");
                    _printerManager.PrintPageModeImageFile(10, 212, imagePath, Dithering.DITHERING_DISABLE);

                    // Specify the QR code
                    // string qrData = "QR サンプル"; // 2-Byte Character Set example
                    string qrData = "QR Sample";      // 1-Byte Character Set example
                    _printerManager.PrintPageModeQRcode(21, 325, qrData, ErrorCorrection.QR_ERROR_CORRECTION_Q, ModuleSize.QR_MODULE_SIZE_2, QrModel.QR_MODEL_2);

                    // Specify the print area in page mode
                    _printerManager.SetPageModeArea(404, 9, 163, 327);

                    // Specify the direction of the print
                    _printerManager.SetPageModeDirection(Direction.DIRECTION_BOTTOM_TO_TOP);

                    // Specify the barcode for EAN13
                    _printerManager.PrintPageModeBarcode(40, 132, BarcodeSymbol.BARCODE_EAN13, "0123456789012", ModuleSize.BARCODE_MODULE_WIDTH_2, 80, HriPosition.HRI_POSITION_ABOVE, CharacterFont.FONT_A);

                    //// Specify the barcode for CODE128 (Data:123456789)
                    //// How to input using the CODE128 Code Set table
                    //// 1. Input the start code of 67H(Code Set A), 68H(Code Set B), or 69H(Code Set C) shown in the table of CODE128 Code Set
                    //// 2. Input any data from 00H to 66H. Multiple data can be input
                    //// 3. Input 67H or more lastly as the stop code
                    ////byte[] data = new byte[] { 0x68, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x67 };

                    //// How to input using the CODE128 Special Code Set table
                    //// 1. Input the start code (START A, START B, or START C) of the code set to be selected shown in the table of CODE128 Special Code
                    //// 2. Input the data in the respective formats
                    ////       Code Set A : Data from 00H to 5FH can be input
                    ////       Code Set B : Data from 20H to 7FH can be input
                    ////       Code Set C : Data from 00H(00) to 63H(99) can be input
                    //byte[] data = new byte[] { 0x7B, 0x42, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39 };
                    //_printerManager.PrintPageModeBarcode(10, 132, BarcodeSymbol.BARCODE_CODE128, data, ModuleSize.BARCODE_MODULE_WIDTH_2, 80, HriPosition.HRI_POSITION_ABOVE, CharacterFont.FONT_A);

                    // Print the page mode and partial cut with pre-feed
                    // If using printer does not have cutter, it will only feed paper
                    _printerManager.PrintPageMode(CuttingMethod.CUT_PARTIAL);

                    // Confirm print finished
                    _printerManager.GetPrinterResponse(PrinterResponseId.PRINTER_RESPONSE_REQUEST, 0, out int[] response);
                    if (response != null && response.Length != 0)
                    {
                        if (response[0] != (int)0x80)
                        {
                            return result = "PageModePrintSample Error";
                        }
                    }
                }
                catch (PrinterException ex)
                {
                    result = $"PageModePrintSample Error : {ex.HResult}";
                }
                finally
                {
                    try
                    {
                        // Exit page mode
                        _printerManager.ExitPageMode();
                    }
                    catch
                    {
                        // An exception error occurs if the page mode is not started or unsupported
                    }
                }

                return result;
            });
        }

        // Sample of StatusCallback
        public Task<string> StatusCallbackSampleAsync(Action<int> callback)
        {
            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    // The event is called when printer status changed
                    _printerManager.SetStatusChangedEventHandler(callback);
                }
                catch (PrinterException ex)
                {
                    result = $"StatusCallbackSample Error : {ex.HResult}";
                }

                return result;
            });
        }

        // Sample of BarcodeScannerCallback
        public Task<string> BarcodeScannerCallbackSampleAsync(Action changedOnline, Action changedOffline, Action<byte[]> scanned)
        {
            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    // The event is called when barcode scanner status changed offline
                    _printerManager.SetBarcodeScannerChangedOnlineEventHandler(changedOnline);

                    // The event is called when barcode scanner status changed online
                    _printerManager.SetBarcodeScannerChangedOfflineEventHandler(changedOffline);

                    // The event is called when barcode scanner scanned barcode
                    _printerManager.SetBarcodeScannerReadDataEventHandler(scanned);
                }
                catch (PrinterException ex)
                {
                    result = $"BarcodeScannerCallbackSample Error : {ex.HResult}";
                }

                return result;
            });
        }

        // Sample using object selection and template showing #1
        public Task<string> BeginPaymentSampleAsync(int total)
        {
            string strTotal = string.Format("$ {0:D1}.{1:D2}", total / 100, total % 100);

            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    _printerManager.SelectTemplate(119, 80);       // Select template (ID=119 : for List Up(En)) with Slide 80
                    // _printerManager.SelectTemplate(120, 80);    // Select template (ID=120 : for List Up(Jp)) with Slide 80

                    _printerManager.SelectTemplateTextObject(0);   // Select Text area 0
                    _printerManager.SetTemplateTextData("Total");
                    _printerManager.SelectTemplateTextObject(1);   // Select Text area 1
                    _printerManager.SetTemplateTextData(strTotal);
                    _printerManager.ShowTemplate(0);               // Display
                }
                catch (PrinterException ex)
                {
                    result = $"DisplaySample Error : {ex.HResult}";
                }

                return result;
            });
        }

        // Sample using object selection and template showing #2
        public Task<string> RegisterItemSampleAsync(int subTotal, string itemName, int price)
        {
            string strSubTotal = string.Format("$ {0:D1}.{1:D2}", subTotal / 100, subTotal % 100);
            string strPrice    = string.Format("$ {0:D1}.{1:D2}", price / 100, price % 100);

            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    _printerManager.SelectTemplateTextObject(1);      // Select Text area 1
                    _printerManager.SetTemplateTextData(strSubTotal);
                    _printerManager.SelectTemplateTextObject(6);      // Select Text area 6
                    _printerManager.SetTemplateTextData(itemName);
                    _printerManager.SelectTemplateTextObject(7);      // Select Text area 7
                    _printerManager.SetTemplateTextData(strPrice);
                    _printerManager.ShowTemplate(0);                  // Display
                }
                catch (PrinterException ex)
                {
                    result = $"DisplaySample Error : {ex.HResult}";
                }

                return result;
            });
        }

        // Sample using object selection and template showing #3
        public Task<string> FinishPaymentSampleAsync(int total, int paid, int change)
        {
            string strTotal  = string.Format("$ {0:D1}.{1:D2}", total / 100, total % 100);
            string strPaid   = string.Format("$ {0:D1}.{1:D2}", paid / 100, paid % 100);
            string strChange = string.Format("$ {0:D1}.{1:D2}", change / 100, change % 100);

            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    _printerManager.SelectTemplate(123, 82);        // Select template (ID=123 : for 3 lines(En)) with Slide 82
                    // _printerManager.SelectTemplate(124, 83);     // Select template (ID=124 : for 3 lines(Jp)) with Slide 83

                    _printerManager.SelectTemplateTextObject(3);    // Select Text area 3
                    _printerManager.SetTemplateTextData(strTotal);
                    _printerManager.SelectTemplateTextObject(4);    // Select Text area 4
                    _printerManager.SetTemplateTextData(strPaid);
                    _printerManager.SelectTemplateTextObject(5);    // Select Text area 5
                    _printerManager.SetTemplateTextData(strChange);

                    _printerManager.ShowTemplate(0);                // Display
                }
                catch (PrinterException ex)
                {
                    result = $"DisplaySample Error : {ex.HResult}";
                }

                return result;
            });
        }

        //	Sample of putting the display on standby
        public Task<string> StandbySampleAsync()
        {
            return Task.Run(() =>
            {
                string result = "";
                try
                {
                    _printerManager.EnterStandbyMode();
                }
                catch (PrinterException ex)
                {
                    result = $"DisplaySample Error : {ex.HResult}";
                }

                return result;
            });
        }

        // Sample of device search
        public async Task<string> SearchDeviceSample(PortType portType)
        {
            string result = "";
            try
            {
                // Search for printer
                var platformSpecifics = DependencyService.Get<IPlatformSpecifics>();
                await platformSpecifics.CheckPermissions(portType);
                _printerManager.StartDiscoveryDevice(DeviceType.TYPE_PRINTER, portType, 5000, Completion);

                //// Search for display (DSP-A01)
                //_printerManager.StartDiscoveryDevice(DeviceType.TYPE_DISPLAY, portType, 5000, Completion);
            }
            catch (PrinterException ex)
            {
                result = $"SearchSample Error : {ex.HResult}";
            }

            return result;
        }

        // Sample of completion delegate of StartDiscoveryDevice
        private void Completion()
        {
            DeviceSearchCompletedEvent?.Invoke(_printerManager.GetFoundDevice());
        }

        // Get SampleLogo.jpg file path.
        // If file does not exist, save file to document folder of application
        private string GetSampleLogoFilePath(string fileName)
        {
            var documentsFolder = System.Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
            string filePath = System.IO.Path.Combine(documentsFolder, fileName);

            if (!System.IO.File.Exists(filePath))
            {
                var assembly = System.Reflection.IntrospectionExtensions.GetTypeInfo(typeof(MainPage)).Assembly;
                using (System.IO.Stream stream = assembly.GetManifestResourceStream($"Sample.Resources.{fileName}"))
                using (System.IO.MemoryStream ms = new System.IO.MemoryStream())
                using (System.IO.FileStream fs = new System.IO.FileStream(filePath, System.IO.FileMode.Create, System.IO.FileAccess.ReadWrite))
                using (System.IO.BinaryWriter writer = new System.IO.BinaryWriter(fs))
                {
                    stream.CopyTo(ms);
                    writer.Write(ms.ToArray());
                }
            }

            return filePath;
        }
    }
}
