﻿using System.Threading.Tasks;
using System.Collections.Generic;

using Android.Bluetooth;
using Android.Content;
using Android.Gms.Common.Apis;
using Android.Gms.Location;
using Android.Net.Wifi;
using Android.OS;
using Android.Provider;

using Xamarin.Forms;
using Xamarin.Essentials;

[assembly: Dependency(typeof(Sample.Droid.PlatformSpecifics))]
namespace Sample.Droid
{
    public class PlatformSpecifics : IPlatformSpecifics
    {
        public object GetApplicationContext()
        {
            return (object)Android.App.Application.Context;
        }

        // Check the permissions required to use the SDK on Android
        public async Task CheckPermissions(SII.SDK.Xamarin.Printer.PortType portType)
        {
            if(portType == SII.SDK.Xamarin.Printer.PortType.PORT_TYPE_BLUETOOTH)
            {
                // Request Bluetooth permissions for Android 12 or higher.
                if (Android.OS.Build.VERSION.SdkInt >= Android.OS.BuildVersionCodes.S)
                {
                    var bluetoothPermissionStatus = await Permissions.CheckStatusAsync<MyBluetoothPermission>();
                    if (bluetoothPermissionStatus != PermissionStatus.Granted)
                    {
                        bluetoothPermissionStatus = await Permissions.RequestAsync<MyBluetoothPermission>();
                        if(bluetoothPermissionStatus != PermissionStatus.Granted)
                        {
                            return;
                        }
                    }
                }

                // If Bluetooth is disabled, enable it.
                var result = await CheckBluetoothEnabled();
                if (!result)
                {
                    return;
                }

                // Enable location permission to discover Bluetooth devices.
                var locationPermissionStatus = await Permissions.CheckStatusAsync<Permissions.LocationWhenInUse>();
                if (locationPermissionStatus != PermissionStatus.Granted)
                {
                    await Permissions.RequestAsync<Permissions.LocationWhenInUse>();
                }
                // for Android 10 or higher
                if (Android.OS.Build.VERSION.SdkInt >= Android.OS.BuildVersionCodes.Q)
                {
                    await CheckLocationRequest();
                }
            }
            else if(portType == SII.SDK.Xamarin.Printer.PortType.PORT_TYPE_TCP)
            {
                // If Wi-Fi is disabled, enable it.
                await CheckWifiEnabled();
            }

            return;
        }

        // If Wi-Fi is disabled, enable it.
        public async Task<bool> CheckWifiEnabled()
        {
            using (WifiManager wifiManager = (WifiManager)Xamarin.Essentials.Platform.CurrentActivity.ApplicationContext.GetSystemService(Context.WifiService))
            {
                if (wifiManager != null && !wifiManager.IsWifiEnabled)
                {
                    var tcs = new TaskCompletionSource<bool>();
                    if (Build.VERSION.SdkInt >= BuildVersionCodes.Q)
                    {
                        MessagingCenter.Subscribe<ActivityResultMessage>(this, ActivityResultMessage.Key, (sender) =>
                        {
                            tcs.SetResult(true);
                        });

                        Intent panelIntent = new Intent(Settings.Panel.ActionWifi);
                        panelIntent.AddFlags(ActivityFlags.NoHistory);
                        Xamarin.Essentials.Platform.CurrentActivity.StartActivityForResult(panelIntent, ActivityResultMessage.WifiEnabledRequestCode);
                    }
                    else
                    {
                        wifiManager.SetWifiEnabled(true);
                        tcs.SetResult(true);
                    }

                    bool result = await tcs.Task;
                    MessagingCenter.Unsubscribe<ActivityResultMessage>(this, ActivityResultMessage.Key);
                    return result;
                }
            }

            return true;
        }

        // If Bluetooth is disabled, enable it.
        public async Task<bool> CheckBluetoothEnabled()
        {
            BluetoothAdapter bluetoothAdapter = BluetoothAdapter.DefaultAdapter;
            if(bluetoothAdapter == null)
            {
                return false;
            }

            if (!bluetoothAdapter.IsEnabled)
            {
                var tcs = new TaskCompletionSource<bool>();
                MessagingCenter.Subscribe<ActivityResultMessage>(this, ActivityResultMessage.Key, (sender) =>
                {
                    tcs.SetResult(true);
                });

                Intent enableBluetoothIntent = new Intent(BluetoothAdapter.ActionRequestEnable);
                Xamarin.Essentials.Platform.CurrentActivity.StartActivityForResult(enableBluetoothIntent, ActivityResultMessage.BluetoothEnabledRequestCode);

                bool result = await tcs.Task;
                MessagingCenter.Unsubscribe<ActivityResultMessage>(this, ActivityResultMessage.Key);
                return result;
            }

            return true;
        }

        // Enable LocationServices to discover Bluetooth devices.
        public async Task<bool> CheckLocationRequest()
        {
            var tcs = new TaskCompletionSource<bool>();
            try
            {
                LocationRequest locationRequest = LocationRequest.Create();
                locationRequest.SetPriority(LocationRequest.PriorityHighAccuracy);
                LocationSettingsRequest.Builder builder = new LocationSettingsRequest.Builder().AddLocationRequest(locationRequest);
                var response = await LocationServices.GetSettingsClient(Xamarin.Essentials.Platform.CurrentActivity).CheckLocationSettingsAsync(builder.Build());
                tcs.SetResult(true);
            }
            catch (ApiException exception)
            {
                switch (exception.StatusCode)
                {
                    case LocationSettingsStatusCodes.ResolutionRequired:
                        MessagingCenter.Subscribe<ActivityResultMessage>(this, ActivityResultMessage.Key, (sender) =>
                        {
                            tcs.SetResult(true);
                        });
                        try
                        {
                            // Request for location information setting
                            ResolvableApiException resolvable = (ResolvableApiException)exception;
                            resolvable.StartResolutionForResult(Xamarin.Essentials.Platform.CurrentActivity, ActivityResultMessage.LocationRequestCode);
                        }
                        catch
                        {
                            tcs.SetResult(false);
                        }
                        
                        break;
                    default:
                        break;
                }

            }

            bool result = await tcs.Task;
            MessagingCenter.Unsubscribe<ActivityResultMessage>(this, ActivityResultMessage.Key);

            return result;
        }
    }
}