﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Xamarin.Forms;

using SII.SDK.Xamarin.Printer;
using System.Collections.ObjectModel;

namespace Sample
{
    public partial class MainPage : ContentPage
    {
        private Function _function;

        public DeviceInfo SelectedDevice { get; set; }

        // Whether status callback starting or not.
        private bool _isStartingStatusCallback = false;

        // Whether barcode scanner callback starting or not.
        private bool _isStartingBarcodeScannerCallback = false;

        public MainPage()
        {
            InitializeComponent();

            _function = new Function();
            _function.DeviceSearchCompletedEvent += this.DeviceSearchCompleted;
        }

        private async void SearchButton_Clicked(object sender, EventArgs e)
        {
            Indicator.IsVisible = true;
            string result = await _function.SearchDeviceSample(GetSelectedPoetType());
            if (result != "")
            {
                Alert("ERROR", result);
                Indicator.IsVisible = false;
            }
        }

        private async void ConnectButton_Clicked(object sender, EventArgs e)
        {
            Indicator.IsVisible = true;

            string result = await _function.ConnectSampleAsync(GetSelectedDeviceModel(), SelectedDevice);
            if (result != "")
            {
                Alert("ERROR", result);
            }
            else
            {
                Alert("INFO", "Connect success");
            }

            Indicator.IsVisible = false;
        }

        private async void DisconnectButton_Clicked(object sender, EventArgs e)
        {
            Indicator.IsVisible = true;

            string result = await _function.DisonnectSampleAsync();
            if (result != "")
            {
                Alert("ERROR", result);
            }
            else
            {
                this.StatusLabel.Text = "-";
                this.ScannerStatusLabel.Text = "-";
                this.ScannedDataLabel.Text = "-";
                Alert("INFO", "Disconnect success");
            }

            Indicator.IsVisible = false;
        }

        private async void StandardSampleButton_Clicked(object sender, EventArgs e)
        {
            Indicator.IsVisible = true;

            string result = await _function.StandardModePrintSampleAsync();
            if (result != "")
            {
                Alert("ERROR", result);
            }
            else
            {
                Alert("INFO", "Standard Mode Sample success");
            }

            Indicator.IsVisible = false;
        }

        private async void PageSampleButton_Clicked(object sender, EventArgs e)
        {
            Indicator.IsVisible = true;

            string result = await _function.PageModePrintSampleAsync();
            if (result != "")
            {
                Alert("ERROR", result);
            }
            else
            {
                Alert("INFO", "Page Mode Sample success");
            }

            Indicator.IsVisible = false;
        }

        private async void StatusCallbackSampleButton_Clicked(object sender, EventArgs e)
        {
            string result;
            if (_isStartingStatusCallback)
            {
                // Cancel callback
                result = await _function.StatusCallbackSampleAsync(null);
                this.StatusLabel.Text = "-";
            }
            else
            {
                // Start callback
                result = await _function.StatusCallbackSampleAsync(this.StatusChanged);
            }

            if (result != "")
            {
                Alert("ERROR", result);
            }
            else
            {
                _isStartingStatusCallback = !_isStartingStatusCallback;
            }
        }

        private async void ScannerCallbackSampleButton_Clicked(object sender, EventArgs e)
        {
            string result;
            if (_isStartingBarcodeScannerCallback)
            {
                // Cancel callback
                result = await _function.BarcodeScannerCallbackSampleAsync(null, null, null);
                this.ScannerStatusLabel.Text = "-";
                this.ScannedDataLabel.Text = "-";
            }
            else
            {
                // Start callback
                result = await _function.BarcodeScannerCallbackSampleAsync(this.ScannerOnline, this.ScannerOffline, this.ScannedData);
            }

            if (result != "")
            {
                Alert("ERROR", result);
            }
            else
            {
                _isStartingBarcodeScannerCallback = !_isStartingBarcodeScannerCallback;
            }
        }

        private async void DisplaySampleButton_Clicked(object sender, EventArgs e)
        {
            Indicator.IsVisible = true;
            try
            {
                _total = 0;
                _itemIndex = 0;

                string result = await _function.BeginPaymentSampleAsync(_total);
                if (result != "")
                {
                    Alert("ERROR", result);
                    return;
                }

                await Task.Delay(1000);

                for (int i = 0; i < 5; i++)
                {
                    PickAnItem();

                    result = await _function.RegisterItemSampleAsync(_total, _itemName, _itemPrice);
                    if (result != "")
                    {
                        Alert("ERROR", result);
                        return;
                    }

                    await Task.Delay(300);
                }

                await Task.Delay(1000);

                CalculateTotal();

                result = await _function.FinishPaymentSampleAsync(_total, _paid, _change);
                if (result != "")
                {
                    Alert("ERROR", result);
                    return;
                }

                await Task.Delay(3000);

                result = await _function.StandbySampleAsync();
                if (result != "")
                {
                    Alert("ERROR", result);
                    return;
                }

                Alert("INFO", "Display Sample success");
            }
            finally
            {
                Indicator.IsVisible = false;
            }
        }

        public void DeviceSearchCompleted(DeviceInfo[] devices)
        {
            Xamarin.Forms.Device.BeginInvokeOnMainThread(async () =>
            {
                Indicator.IsVisible = false;

                if (devices != null && devices.Length != 0)
                {
                    Dictionary<string, DeviceInfo> deviceList = new Dictionary<string, DeviceInfo>();
                    foreach (var device in devices)
                    {
                        string key = "";
                        switch (device.PortType)
                        {
                            case PortType.PORT_TYPE_BLUETOOTH:
                                key = $"{device.DeviceName}({device.BluetoothAddress})";
                                break;
                            case PortType.PORT_TYPE_TCP:
                                key = $"{device.IpAddress}({device.MacAddress})";
                                break;
                            case PortType.PORT_TYPE_USB:
                                key = $"{device.DeviceName}({device.DevicePath})";
                                break;
                            default:
                                continue;
                        }

                        deviceList.Add(key, device);
                    }
                    string selectDevice = await DisplayActionSheet("Select Device", "Cancel", null, deviceList.Keys.ToArray());

                    if (deviceList.ContainsKey(selectDevice))
                    {
                        SelectedDevice = deviceList[selectDevice];
                        SelectedDeviceLabel.Text = selectDevice;
                    }
                }
                else
                {
                    Alert("ERROR", "Device is not found.");
                }
            });
        }

        // Sample of statusChangedEventHandler delegate of SetStatusChangedEventHandler
        public void StatusChanged(int status)
        {
            Xamarin.Forms.Device.BeginInvokeOnMainThread(() =>
            {
                StatusLabel.Text = $"0x{status:X8}";
            });
        }

        // Sample of barcodeScannerChangedOnlineEventHandler delegate of SetBarcodeScannerChangedOnlineEventHandler
        public void ScannerOffline()
        {
            Xamarin.Forms.Device.BeginInvokeOnMainThread(() =>
            {
                ScannerStatusLabel.Text = "Offline";
            });
        }

        // Sample of barcodeScannerChangedOfflineEventHandler delegate of SetBarcodeScannerChangedOfflineEventHandler
        public void ScannerOnline()
        {
            Xamarin.Forms.Device.BeginInvokeOnMainThread(() =>
            {
                ScannerStatusLabel.Text = "Online";
            });
        }

        // Sample of barcodeScannerReadDataEventHandler delegate of SetBarcodeScannerReadDataEventHandler
        public void ScannedData(byte[] scannedData)
        {
            Xamarin.Forms.Device.BeginInvokeOnMainThread(() =>
            {
                try
                {
                    string data;
                    if (this.EncodeSwitch.IsToggled)
                    {
                        data = Encoding.GetEncoding("shift_jis").GetString(scannedData);
                    }
                    else
                    {
                        data = BitConverter.ToString(scannedData).Replace("-", string.Empty);
                    }
                    ScannedDataLabel.Text = data;
                }
                catch (Exception)
                {
                    ScannedDataLabel.Text = "Encoding failed.";
                }
            });
        }

        private DeviceModel GetSelectedDeviceModel()
        {
            switch (DeviceModelPicker.SelectedIndex)
            {
                case 0:
                    return DeviceModel.DEVICE_MODEL_RP_E10;
                case 1:
                    return DeviceModel.DEVICE_MODEL_RP_D10;
                case 2:
                    return DeviceModel.DEVICE_MODEL_MP_B20;
                case 3:
                    return DeviceModel.DEVICE_MODEL_MP_B30;
                case 4:
                    // Includes RP-G10
                    return DeviceModel.DEVICE_MODEL_RP_F10;
                case 5:
                    return DeviceModel.DEVICE_MODEL_DSP_A01;
                case 6:
                    return DeviceModel.DEVICE_MODEL_MP_B30L;
                case 7:
                    return DeviceModel.DEVICE_MODEL_SLP720RT;
                default:
                    return DeviceModel.DEVICE_MODEL_RP_E10;
            }
        }

        private PortType GetSelectedPoetType()
        {
            switch (SearchTypePicker.SelectedIndex)
            {
                case 0:
                    return PortType.PORT_TYPE_BLUETOOTH;
                case 1:
                    return PortType.PORT_TYPE_TCP;
                case 2:
                    return PortType.PORT_TYPE_USB;
                default:
                    return PortType.PORT_TYPE_BLUETOOTH;
            }
        }

        #region Used in the display sample.
        private int _total = 0;
        private int _paid = 0;
        private int _change = 0;
        private int _itemIndex = 0;
        private string _itemName = "";
        private int _itemPrice = 0;

        private string[] _itemNames = new string[5]
        {
            "BROCCOLI 0.808kg ($5.99/kg)",
            "BANANA 0.442kg ($2.99/kg)",
            "POTATO 1.328kg ($2.99/kg)",
            "GRAPES 1.174kg ($5.99/kg)",
            "PEAS 0.218kg ($4.99/kg)"
        };

        private int[] _itemPrices = new int[5]
        {
            132,
            186,
            391,
            116,
            100
        };

        private void PickAnItem()
        {
            _itemName = _itemNames[_itemIndex];     // Pick an item name
            _itemPrice = _itemPrices[_itemIndex];   // Pick an item price	
            _itemIndex = (_itemIndex + 1) % 5;
            _total += _itemPrice;                   // Calculate total
        }

        private void CalculateTotal()
        {
            _paid = ((_total + 999) / 1000) * 1000;
            _change = _paid - _total;
        }
        #endregion

        public void Alert(string title, string message)
        {
            Xamarin.Forms.Device.BeginInvokeOnMainThread(() =>
            {
                _ = DisplayAlert(title, message, "OK");
            });
        }
    }
}
